/*******************************************************************************
  tagTest.c:  This is a march test of the tag address bits.  Caches of course
  reflect content of other memories, and for this test, that memory has to be
  a block of memory...
  * whose base address can be remapped at will, and
  * is at least as big as the cache data RAM.
  This code in itself does not require that it be any particular such moveable
  RAM, but uses calls to functions defined elsewhere to move it and to access
  it in certain important ways.  They are defined elsewhere so that this RAM 
  can be one kind of RAM can be application-specific.
  
  Those "important ways" relate to the fact that we have to carefully control 
  the cache's image of that RAM.  The cache is generally kept locked (frozen),
  and in fact disabled, throughout the test, except for certain functions that 
  momentarily unlock and enable the cache to cause the tags to be modified.
  Conceptually, locking it is sufficient, but it doesn't complicate things much
  to disable it too, so we may as well.
*******************************************************************************/
#include "src/init/m5282evb.h"
#include "environment.h"
#include "moveRam.h"  /* header for the moveable RAM field*/
#include "march.h"

#define KRAMBase 0x20000000
#define INNOCUOUSADDR 0x55555000
   /* This is just any innocuous address to map the moveable RAM to so that we 
      can initially fill it with HITVALUE. */

extern UINT32 asmAddr;

/********************************************************
 * fillTags() fills the tags with a stated pattern.
 ********************************************************/ 
void fillTags (UINT32 pattern) {
   UINT32
      *addr,  /* just a generic pointer*/
      *frontDoorAddr,  /* address where the RAM gets moved to*/
      data, /* Dummy read destination*/
      offset; /*Offset from base of real KRAM*/

   moveRam((UINT32*)(pattern)); 

   COMMENT("Fill the KRAM with 2K of HITVALUE\n");
   for (addr = (UINT32*)(pattern);  addr < (UINT32*)(pattern) + (SIZE / 4); *(addr++) = HITVALUE) {} 

   ASM(move.w #0x2000,sr); /* Set supervisor mode*/

   /*Disable and invalidate the entire cache*/
   ASM(move.l #0x01800103,d0);  /* disable-and-invalidate cache CACR value*/
   ASM(movec d0,cacr);
   /*Enable the cache (as data only)*/
   ASM(move.l #0x80800103,d0);  /* enable-and-unlock-cache CACR value*/
   ASM(movec d0,cacr);
   frontDoorAddr = (UINT32*)((KRAMBase+(pattern&0x0000ffff)) );

   COMMENT("Fill the cache with HITVALUE");
   for (addr = (UINT32*)(pattern);  addr < (UINT32*)(pattern) + (SIZE / 4); *(addr++)) {data=*addr;} 
   COMMENT("Replace RAM contents with MISSVALUE through front door, cache should still read HITVALUE\n");
   for (addr = frontDoorAddr ;  addr < frontDoorAddr + (SIZE / 4); *(addr++) = MISSVALUE) {} 
   return;
}

/********************************************************************************
 * tagBad() verifies that a tag produces a cache hit, returning one (true)
 *  if so, or zero (false) if not.  Whether or not the tag was written correctly
 *  is easy to determine:  
 *  1.  Disable the cache (it probably already will be, but just to be sure...)
 *  2.  Write MISSVALUE around the cache to the first longword in line it tags, 
 *  3.  Re-enable the cache and read from that same place.
 *  4.  If the tag is written wrong, we'll get a miss, as evidenced by reading
 *      MISSVALUE from memory rather than HITVALUE from cache.
 *********************************************************************************/
int tagBad (int tagNo, UINT32 pattern) {
   UINT32
      *movRamAddr,  /* address where the RAM gets moved to*/
      *frontDoorAddr,  /* address where the RAM gets moved to*/
	   readValue;  /* what we read from the moveable RAM to check caching*/
   
   movRamAddr = (UINT32*) (pattern & ~(TAGARRAYSIZE - 1));
   asmAddr = (UINT32)(&movRamAddr[tagNo * 4]);

   /* Hopefully read the cache instead of memory:*/
   readValue = movRamAddr[tagNo * 4];  /* *4 because each tag covers 4 longwords*/
   if (readValue != HITVALUE)
      return 1;  // false, fail*/
   
   /* Make sure that the valid bit is not stuck high:*/
      ASM(move.l asmAddr,a0);  /* Invalidate cache line */
      ASM(cpushl #0x3,(a0));  /* push and invalidate*/

      /* Reread that address.  If tag is falsely valid, it will get the HITVALUE*/
      readValue = movRamAddr[tagNo * 4];  /* *4 because each tag covers 4 longwords*/
      if (readValue != MISSVALUE)
         return 2;  // false, fail*/
   /* Restore HITVALUE to the CACHE:*/
   movRamAddr[tagNo * 4] = HITVALUE;
   readValue=movRamAddr[tagNo*4];
   frontDoorAddr = (UINT32*)((KRAMBase+(pattern&0x0000ffff)) );
   frontDoorAddr[tagNo * 4] = MISSVALUE;
   /* Report our results to the caller:*/
   return 0; /*true pass*/
}
/*******************************************************************************
  The following functions do the basic march-test actions:  Filling the tags
  with an initial pattern, checking whether a tag was written correctly or not,
  and writing a value to a tag.
*******************************************************************************/
/* tagMarchPat() runs one single pattern of a tag march test.*/
int tagMarchPat (
   UINT32 pattern  /* the longword pattern to test with on this pass*/
) 
   {
   UINT32 masked_pattern; /*Pattern with memory mask to prevent overwriting code*/
   int tagNo;  /* which tag we're currently "dealing with"*/
   UINT32* aPtr;  /* used just to fill the moveable RAM*/
   masked_pattern = pattern&LOWER32K;
   /* Flood-fill the tags with the pattern:*/
   fillTags (masked_pattern);

   /* One by one, read/check:*/
   for (tagNo = 0;  tagNo < 128;  tagNo++) {
      if (tagBad(tagNo, masked_pattern))  /* check for pattern*/
         return FAILFILL;  // fail
   }
   return MARCHPASS;  /* if we make it this far, we passed the test.*/
}